/* The routines in this file implement a doubly linked list.         */
/*                                                                   */
/* NOTE: Be careful here. The head and tail members of an empty list */
/*       point to the list rather than containing NULL as is often   */
/*       done elsehwere. This eliminates the overhead of checking for*/
/*       an empty list case in several of these routines. It also    */
/*       allows the node removal functions to be streamlined as they */
/*       don't have to worry about checking for an empty list and    */
/*       then having to set both pointers in the list header to NULL.*/

#include <stdlib.h>
#include "lists.h"


/* Create a new node */
static Node *NodeCreate(Data *data)
{
Node *new_node;

	new_node = (Node *)malloc(sizeof(Node));
	if (new_node == NULL)
		return NULL;

	new_node->data = data;
	return new_node;
}


/* Add a node as a successor to a specified node */
void NodeAdd(Node *prev_node, Node *new_node)
{
	if (prev_node == NULL)
		return;

	new_node->next = prev_node->next;
	new_node->prev = prev_node;
	prev_node->next = new_node;
	new_node->next->prev = new_node;
}


/* Create a new node and add it as a successor to a specified node */
Node *NodeAddNew(Node *prev_node, Data *data)
{
Node *new_node;

	if (prev_node == NULL)
		return NULL;

	new_node = NodeCreate(data);
	if (new_node == NULL)
		return NULL;

	NodeAdd(prev_node, new_node);

	return new_node;
}


/* Remove a node from a list */
Node *NodeRemove(Node *node)
{
	if (node == NULL)
		return node;

	node->prev->next = node->next;
	node->next->prev = node->prev;

	return node;
}


/* Removes a node from the list its in and then frees both */
/* the data pointed to by the node and the node itself.    */
void NodeDestroy(Node *node)
{
Data *data;

	if (node == NULL)
		return;

	data = node->data;
	if (data != NULL)
		free(data);

	free(node);
}


/* Searches a list for the node containing */
/* the pointer to the specified data area. */
Node *NodeFind(List *list, Data *data)
{
Node *node;

	if (list == NULL)
		return NULL;

	node = NodeHead(list);
	while (node != (Node *)list)
	{
		if (node->data == data)
			return node;

		node = NodeNext(node);
	}

	return NULL;
}


/* Create a new list */
List *ListCreate(void)
{
List *new_list;

	new_list = (List *)malloc(sizeof(List));
	if (new_list == NULL)
		return NULL;

	new_list->head = (Node *)new_list;
	new_list->tail = (Node *)new_list;

	return new_list;
}


/* Removes all nodes from a list */
void ListRemAll(List *list)
{
Node *node;

	if (list == NULL)
		return;

	while (NodeTail(list) != (Node *)list)
	{
		node = NodeRemTail(list);
		NodeDestroy(node);
	}
}


/* Destroys all nodes in the list and then */
/* finally frees the list header itself.   */
void ListDestroy(List *list)
{
	if (list == NULL)
		return;

	ListRemAll(list);
	free(list);
}
