#include "fix.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <assert.h>

#include "../telnet.h"
#include "telnet-fsm.h"

/* get the function declarations for tnmatrix[] */
#include "telnet.h"

/*** Telnet--Socket Finite State Machine states ***/
#define TSDATA		0	/* normal state 		*/
#define TSIAC		1	/* seen IAC				*/
#define TSWOPT		2	/* seen IAC-WILL/WONT	*/
#define TSDOPT		3	/* seen IAC-DO/DONT		*/
#define TSSUBNEG	4	/* seen IAC SB			*/
#define TSSUBIAC	5	/* seen IAC SB .. IAC		*/
#define TSSUBERR	6	/* too many chars in IAC SB */
#define TSERRIAC	7	/* seen IAC during TSSUBERR */
#define TSTOTAL		8	/* TOTAL NUMBER OF STATES	*/

/*** Telnet sub-negotiation Finite State Machine states ***/
#define TSB_START		0	/* seen IAC SB              */
#define TSB_LINEMODE	1	/* seen IAC SB LINEMODE     */
#define TSB_LM_MODE		2	/* seen IAC SB LINEMODE MODE*/
#define TSB_LM_WWDD		3	/* seen IAC SB LM WILL/WONT/DO/DONT */
#define TSB_LM_SLC		4	/* seen IAC SB LINEMODE SLC */
#define TSB_NAWS		5	/* seen IAC SB NAWS         */
#define TSB_DATA		6	/* getting sub-negotiation data */
#define TSB_FLUSH		7	/* flushing data after error	*/
#define TSB_TOTAL		8	/* TOTAL NUMBER OF STATES   */

static int cfsm_dopt(struct tnstat*, unsigned char);
static int cfsm_wopt(struct tnstat*, unsigned char);

static int cfsm_nop(struct tnstat*, unsigned char);
static int cfsm_unsup(struct tnstat*, unsigned char);
static int cfsm_dm(struct tnstat*, unsigned char);
static int cfsm_store(struct tnstat*, unsigned char);
static int cfsm_cpy(struct tnstat*, unsigned char);

static int cfsm_subbegin(struct tnstat*, unsigned char);
static int cfsm_subneg(struct tnstat*, unsigned char);
static int cfsm_subend(struct tnstat*, unsigned char);

static int sb_error(struct tnstat*, unsigned char);
static int sb_unsup(struct tnstat*, unsigned char);
static void sb_linemode(struct tnstat *ptr);
static void sb_lm_mode(struct tnstat *ptr);
static void sb_lm_fwmask(struct tnstat *ptr);
static void sb_lm_slc(struct tnstat *ptr);
static void sb_naws(struct tnstat *ptr);

static void tnthemenable(struct tnstat*, unsigned char);
static void tnthemdisable(struct tnstat*, unsigned char);
static void tnusenable(struct tnstat*, unsigned char);
static void tnusdisable(struct tnstat*, unsigned char);

#ifdef	DEBUG_TELNET
static char *tncomis(unsigned char);
#ifndef	NEW_SOCKET_IO
#define write_to_socket(user, str, length)  write(user->sock, str, length)
#else
static int perform_socket_write(struct tnstat *ptr, unsigned char *str, size_t length);
static int write_to_socket(struct tnstat *ptr, unsigned char *txt, int bytes_to_write);
#endif
void debug_write(struct tnstat *ptr, char *buff, int size);

char tndmess[120];				 /* Holds debugging message */
#endif


struct fsm {
	unsigned char state;	/* current state    */
	short int input;		/* input character  */
	unsigned char next;		/* next state       */
	int (*function)(struct tnstat *, unsigned char);	/* function to call */
};

struct fsm tnmatrix[] = {
/*	+----------+------------+----------------+---------------+	*/
/*	| State	   |	Input	|	Next State   |	Function     |	*/
/*	+----------+------------+----------------+---------------+	*/
	{ TSDATA,	TCIAC,		TSIAC,		NULL		},
	{ TSDATA,	ANYCHAR,	TSDATA,		cfsm_cpy	},
	{ TSIAC,	TCIAC,		TSDATA,		cfsm_cpy	},
	{ TSIAC,	TCSB,		TSSUBNEG,	cfsm_subbegin},
	/* telnet commands */
	{ TSIAC,	TCNOP,		TSDATA,		cfsm_nop	},
	{ TSIAC,	TCDM,		TSDATA,		cfsm_dm		},
	/* option negotiation */
	{ TSIAC,	TCWILL,		TSWOPT,		cfsm_store	},
	{ TSIAC,	TCWONT,		TSWOPT,		cfsm_store	},
	{ TSIAC,	TCDO,		TSDOPT,		cfsm_store	},
	{ TSIAC,	TCDONT,		TSDOPT,		cfsm_store	},
	/* unknown command */
	{ TSIAC,	ANYCHAR,	TSDATA,		cfsm_unsup	},
	/* option sub-negotiation */
	{ TSSUBNEG,	TCIAC,		TSSUBIAC,	NULL		},
	{ TSSUBNEG, ANYCHAR,    TSSUBNEG,   cfsm_subneg },
	/* escaped sub-negotiation characters */
	{ TSSUBIAC,	TCSE,		TSDATA,		cfsm_subend	},
	{ TSSUBIAC,	ANYCHAR,	TSSUBNEG,	cfsm_subneg	},
	/* will-wont */
	{ TSWOPT,	ANYCHAR,	TSDATA,		cfsm_wopt	},
	/* do-dont */
	{ TSDOPT,	ANYCHAR,	TSDATA,		cfsm_dopt	},
	/* miracle */
	{ FSINVALID,ANYCHAR,	FSINVALID,	NULL		}
};


/* The following matrix is use by the cfsm_subneg() routine	*/
/* and is called only when indicated by the table above.	*/
/* Handling of escaped IAC codes is not needed here as this	*/
/* is handled by the above matrix used before reaching here.*/
/* This matrix doesn't need to show a return to the start	*/
/* state as that too is handled by the higher level matrix.	*/

struct fsm tnsbmatrix[] = {
/*	+----------+------------+----------------+---------------+	*/
/*	| State	   |	Input	|	Next State   |	Function     |	*/
/*	+----------+------------+----------------+---------------+	*/
	{ TSB_START,	TNLineMode,	TSB_LINEMODE,	NULL	},
	{ TSB_START,	TNNegotiateAboutWindowSize,	TSB_NAWS,	NULL},
	{ TSB_START,	ANYCHAR,	TSB_FLUSH,		sb_unsup},

	{ TSB_LINEMODE, LM_MODE,	TSB_LM_MODE,	NULL	},
	{ TSB_LINEMODE,	LM_SLC,		TSB_LM_SLC,		NULL	},
	{ TSB_LINEMODE,	TCWILL,		TSB_LM_WWDD,	NULL	},
	{ TSB_LINEMODE,	TCWONT,		TSB_LM_WWDD,	NULL	},
	{ TSB_LINEMODE,	TCDO,		TSB_LM_WWDD,	NULL	},
	{ TSB_LINEMODE,	TCDONT,		TSB_LM_WWDD,	NULL	},
	{ TSB_LINEMODE,	ANYCHAR,	TSB_FLUSH,		sb_error},

	{ TSB_LM_MODE,	ANYCHAR,	TSB_DATA,		NULL	},

	{ TSB_LM_WWDD,	LM_FORWARDMASK,	TSB_DATA,	NULL	},
	{ TSB_LM_WWDD,	ANYCHAR,		TSB_FLUSH,	sb_error},

	{ TSB_LM_SLC,	ANYCHAR,	TSB_DATA,		NULL	},

	{ TSB_NAWS,		ANYCHAR,	TSB_DATA,		NULL	},

	{ TSB_DATA,		ANYCHAR,	TSB_DATA,		NULL	},
	{ TSB_FLUSH,	ANYCHAR,	TSB_FLUSH,		NULL	},

	{ FSINVALID,	ANYCHAR,	FSINVALID,		NULL	}
};

ttable_ptr tnttable;
ttable_ptr tnsbttable;
char *tndesc[256];


/* returns string description of an option */
char *tnoptis(unsigned char c)
{
	static char retstr[40];

	if (tndesc[c] == NULL)
	{
		sprintf(retstr, "Unknown Option (%i)", c);
		return retstr;
	}
	return tndesc[c];
}


#ifdef	DEBUG_TELNET
/* returns description of a telnet command */
static char *tncomis(unsigned char c)
{
	static char desc[5];

	switch (c)
	{
	case TCSB:
		strcpy(desc, "SB");
		break;
	case TCNOP:
		strcpy(desc, "NOP");
		break;
	case TCDM:
		strcpy(desc, "DM");
		break;
	case TCBRK:
		strcpy(desc, "BRK");
		break;
	case TCIP:
		strcpy(desc, "IP");
		break;
	case TCAO:
		strcpy(desc, "AO");
		break;
	case TCAYT:
		strcpy(desc, "AYT");
		break;
	case TCEC:
		strcpy(desc, "EC");
		break;
	case TCEL:
		strcpy(desc, "EL");
		break;
	case TCGA:
		strcpy(desc, "GA");
		break;
	case TCSE:
		strcpy(desc, "SE");
		break;
	case TCWILL:
		strcpy(desc, "WILL");
		break;
	case TCWONT:
		strcpy(desc, "WONT");
		break;
	case TCDO:
		strcpy(desc, "DO");
		break;
	case TCDONT:
		strcpy(desc, "DONT");
		break;
	case TCIAC:
		strcpy(desc, "IAC");
		break;
	default:
		sprintf(desc, "x%x", c);
		break;
	}
	return desc;
}
#endif


/*
** This routine generates a transition table for the compact fsm pointed to
** by matrix. It mallocs the memory and returns a pointer to the table. If
** it couldn't malloc memory, a NULL pointer is returned.
*/
static ttable_ptr
make_ttable(struct fsm *matrix, int noofstates)
{
	ttable_ptr ptr;
	int i, s, c;

	if ((ptr = (ttable_ptr) malloc(sizeof(ttable) * noofstates)) == NULL)
		return NULL;

	/* clear all entries first */
	for (s = 0; s < noofstates; ++s)
		for (c = 0; c < NCHARS; ++c)
			ptr[s][c] = TINVALID;

	/* fill in the expanded state transition table */
	for (i = 0; matrix[i].state != FSINVALID; ++i)
	{
		s = matrix[i].state;
		if (matrix[i].input != ANYCHAR)
			ptr[s][matrix[i].input] = i;
		else
		{
			for (c = 0; c < NCHARS; ++c)
				if (ptr[s][c] == TINVALID)
					ptr[s][c] = i;
		}
	}
	/* now fix any entries not yet done */
	for (s = 0; s < noofstates; ++s)
		for (c = 0; c < NCHARS; ++c)
			if (ptr[s][c] == TINVALID)
				ptr[s][c] = i;	 /* i was set at end of previous loop */

	return ptr;
}


/* returns 1 on failure */
int init_telnet(void)
{
	int i;

	if ((tnttable = make_ttable(tnmatrix, TSTOTAL)) == NULL)
		return 1;

	if ((tnsbttable = make_ttable(tnsbmatrix, TSB_TOTAL)) == NULL)
	{
		free(tnttable);
		return 1;
	}

	for (i = 0; i < NCHARS; i++)
		tndesc[i] = NULL;

	tndesc[0] = "Binary Transmission";
	tndesc[1] = "Echo";
	tndesc[2] = "Reconnection";
	tndesc[3] = "Supress Go Ahead";
	tndesc[4] = "Approx Message Size Negotiation";
	tndesc[5] = "Status";
	tndesc[6] = "Timing Mark";
	tndesc[7] = "Remote Controlled Trans and Echo";
	tndesc[8] = "Output Line Width";
	tndesc[9] = "Output Page Size";
	tndesc[10] = "Output Carriage-Return Disposition";
	tndesc[11] = "Output Horizontal Tab Stops";
	tndesc[12] = "Output Horizontal Tab Disposition";
	tndesc[13] = "Output Formfeed Disposition";
	tndesc[14] = "Output Vertical Tab Stops";
	tndesc[15] = "Output Vertical Tab Disposition";
	tndesc[16] = "Output Linefeed Disposition";
	tndesc[17] = "Extended ASCII";
	tndesc[18] = "Logout";
	tndesc[19] = "Byte Macro";
	tndesc[20] = "Data Entry Terminal";
	tndesc[21] = "SUPDUP";		 /* CHECK THIS ONE OUT!!!! */
	tndesc[22] = "SUPDUP Output";
	tndesc[23] = "Send Location";
	tndesc[24] = "Terminal Type";
	tndesc[25] = "End of Record";
	tndesc[26] = "TACAS User Identification";
	tndesc[27] = "Output Marking";
	tndesc[28] = "Terminal Location Number";
	tndesc[29] = "Telnet 3720 Regime";
	tndesc[30] = "X.3 PAD";
	tndesc[31] = "Negotiate About Window Size";
	tndesc[32] = "Terminal Speed";
	tndesc[33] = "Remote Flow Control";
	tndesc[34] = "Linemode";
	tndesc[35] = "X Display Location";
	tndesc[36] = "Environment Option";
	tndesc[37] = "Authentication Option";
	tndesc[38] = "Encryption Option";
	tndesc[39] = "New Environment Option";
	tndesc[40] = "TN3270E";
	tndesc[255] = "Extended-Options-List";

	return 0;
}


void cleanup_telnet(void)
{
	if (tnttable != NULL)
	{
		free(tnttable);
		tnttable = NULL;
	}
	if (tnsbttable != NULL)
	{
		free(tnsbttable);
		tnsbttable = NULL;
	}
}


/* Receiving telnet data */
void tndata(struct tnstat *ptr, unsigned char c)
{
	struct fsm *p;
	int state;
	int error;

	state = tnttable[ptr->state][c];
	p = &tnmatrix[state];
	if (p->function == NULL)
		ptr->state = p->next;
	else
	{
		error = (p->function) (ptr, c);
		if (!error)
			ptr->state = p->next;
	}
}


#if defined(__cplusplus) && !defined(DEBUG_TELNET)
static int cfsm_nop(struct tnstat *, unsigned char)
#else
static int cfsm_nop(struct tnstat *ptr, unsigned char c)
#endif
{
#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: Received %s]", PROGNAME, tncomis(c));
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif
	return 0;
}


#if defined(__cplusplus) && !defined(DEBUG_TELNET)
static int cfsm_unsup(struct tnstat *, unsigned char)
#else
static int cfsm_unsup(struct tnstat *ptr, unsigned char c)
#endif
{
#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= WARNINGS)
	{
		sprintf(tndmess, "[%s: Unsupported telnet command '%s']", PROGNAME,
				tncomis(c));
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif
	return 0;
}


static int cfsm_cpy(struct tnstat *ptr, unsigned char c)
{
	unsigned char last;
	int goteol = 0;

	if (ptr->synch)
		return 0;

	last = ptr->lastchar;
	ptr->lastchar = c;			 /* update the last character input thing */

	/* this case statement handles older telnet clients that may not use  */
	/* the correct EOL characters (only if EOELBUFFER is set)     */
	if (ptr->eolbuff)
		switch (c)
		{
		case '\n':
			if (last == '\r')
				return 0;		 /* ignore */
			else
				goteol = 1;
			break;
		case '\r':
			goteol = 1;
			break;
		case '\0':
			if (last == '\r')
				return 0;		 /* ignore */
			else
				goteol = 1;
			break;
		default:
			break;
		}

	if (ptr->waitcr == 1 && !goteol)
		return 0;

	if (goteol)
	{
		if (tngetopt(ptr, TNEcho, WEARE))
			write_to_socket(ptr, (unsigned char *)"\r\n", (size_t) 2);
		if (ptr->waitcr == 1)
			ptr->waitcr = 0;
		else
		{
			ptr->buff[ptr->buffpos++] = '\0';
			if (ptr->parser != NULL)
				(ptr->parser) (ptr->buff);
			ptr->buffpos = 0;
			ptr->buff[0] = '\0';
		}
		return 0;
	}

	if (ptr->buffpos + 2 > (int) sizeof(ptr->buff))
	{							 /* buffer full */
		if (ptr->eolbuff)
		{
			ptr->buff[ptr->buffpos++] = '\0';
			ptr->waitcr = 1;
		}
		if (ptr->parser != NULL)
			(ptr->parser) (ptr->buff);
		ptr->buffpos = 0;
		ptr->buff[0] = '\0';
		return 0;
	}

	if (!ptr->filter)
	{
		if (tngetopt(ptr, TNEcho, WEARE) && ptr->echochars)
			write_to_socket(ptr, &c, (size_t) 1);
		ptr->buff[ptr->buffpos++] = c;
		return 0;
	}

	/* naughty way to check for backspace/delete */
	if (c == 8 || c == 127)
	{
		if (ptr->buffpos > 0)
		{
			if (tngetopt(ptr, TNEcho, WEARE) && ptr->echochars)
				write_to_socket(ptr, (unsigned char *)"\b \b", (size_t) 3);
			--ptr->buffpos;
		}
		return 0;
	}

	if (isprint(c))
	{
		if (tngetopt(ptr, TNEcho, WEARE) && ptr->echochars)
			write_to_socket(ptr, &c, (size_t) 1);
		ptr->buff[ptr->buffpos++] = c;
	}
#ifdef DEBUG_TELNET
	else if (ptr->debuglevel >= WARNINGS)
	{
		sprintf(tndmess, "[%s: Ignored character 'x%x']", PROGNAME, c);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	return 0;
}


static int cfsm_dm(struct tnstat *ptr, unsigned char c)
{
	if (ptr->synch == 1)
	{
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= VERBOSE)
		{
			sprintf(tndmess, "[%s: END SYNCH]", PROGNAME);
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		ptr->synch = 0;
	}
	return 0;
}


static int cfsm_store(struct tnstat *ptr, unsigned char c)
{
	ptr->store = c;
	return 0;
}


static int cfsm_wopt(struct tnstat *ptr, unsigned char c)
{
	unsigned char o;
	unsigned char buff[3];
	struct tnopt *opt;

	opt = &(ptr->opt[c]);
	o = ptr->store;
	buff[0] = TCIAC;
	buff[1] = 0;
	buff[2] = c;

#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: Received IAC %s '%s']",
				PROGNAME, tncomis(o), tnoptis(c));
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif
	if (o == TCWILL)
		switch (opt->them)
		{
		case NO:
			/* our choice */
			if (opt->theyshould)
			{
				opt->them = YES;
				buff[1] = TCDO;
				write_to_socket(ptr, buff, 3);
				if (ptr->changeopt != NULL)
					(ptr->changeopt) (ptr, THEYARE, c);
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I let you enable '%s']", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			else
			{
				buff[1] = TCDONT;
				write_to_socket(ptr, buff, 3);
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I didn`t let you enable '%s']", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			break;
		case YES:
			/* already enabled */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= WARNINGS)
			{
				sprintf(tndmess, "[%s: You already have '%s' enabled]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			break;
		case WANTNO:
			/* ERROR: DONT answered by WILL */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= ERRORS)
			{
				sprintf(tndmess, "[%s: DONT answered by WILL for option '%s']", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			if (opt->themq == EMPTY)
			{
				opt->them = NO;
				if (ptr->changeopt != NULL)
					(ptr->changeopt) (ptr, THEYARENT, c);
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I didn`t let you enable '%s']", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			else
			{
				opt->them = YES;
				opt->themq = EMPTY;
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I let you enable '%s']", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			break;
		case WANTYES:
			if (ptr->changeopt != NULL)
				(ptr->changeopt) (ptr, THEYARE, c);
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You enabled '%s' for me]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			if (opt->themq == EMPTY)
				opt->them = YES;
			else
			{
				opt->them = WANTNO;
				opt->themq = EMPTY;
				buff[1] = TCDONT;
				write_to_socket(ptr, buff, 3);
				break;
			}
		}						 /* end switch */
	else if (o == TCWONT)
		switch (opt->them)
		{
		case NO:
			/* already disabled */
			break;
		case YES:
			opt->them = NO;
			buff[1] = TCDONT;
			write_to_socket(ptr, buff, 3);
			if (ptr->changeopt != NULL)
				(ptr->changeopt) (ptr, THEYARENT, c);
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You disabled '%s']", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			break;
		case WANTNO:
			if (ptr->changeopt != NULL)
				(ptr->changeopt) (ptr, THEYARENT, c);
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You disabled '%s' for me]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			if (opt->themq == EMPTY)
			{
				opt->them = NO;
			}
			else
			{
				opt->them = WANTYES;
				opt->themq = EMPTY;
				buff[1] = TCDO;
				write_to_socket(ptr, buff, 3);
			}
			break;
		case WANTYES:
			/* a length-two queue could save a potential request here... */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You wouldn`t enable '%s' for me]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			opt->them = NO;
			if (opt->themq == OPPOSITE)
				opt->themq = EMPTY;
			break;
		}						 /* end switch */
	return 0;
}


static int cfsm_dopt(struct tnstat *ptr, unsigned char c)
{
	unsigned char o;
	unsigned char buff[3];
	struct tnopt *opt;

	opt = &(ptr->opt[c]);
	o = ptr->store;
	buff[0] = TCIAC;
	buff[1] = 0;
	buff[2] = c;

#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: Received IAC %s '%s']", PROGNAME, tncomis(o),
				tnoptis(c));
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif
	if (o == TCDO)
		switch (opt->us)
		{
		case NO:
			/* our choice */
			if (opt->weshould)
			{
				opt->us = YES;
				buff[1] = TCWILL;
				write_to_socket(ptr, buff, 3);
				if (ptr->changeopt != NULL)
					(ptr->changeopt) (ptr, WEARE, c);
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I enabled '%s' for you]", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			else
			{
				buff[1] = TCWONT;
				write_to_socket(ptr, buff, 3);
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I wouldn't enable '%s' for you]", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			break;
		case YES:
			/* already enabled */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= WARNINGS)
			{
				sprintf(tndmess, "[%s: I already have '%s' enabled]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			break;
		case WANTNO:
			/* ERROR: WONT answered by DO */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= ERRORS)
			{
				sprintf(tndmess, "[%s: WONT answered by DO for option '%s']", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			if (opt->usq == EMPTY)
			{
				opt->us = NO;
				if (ptr->changeopt != NULL)
					(ptr->changeopt) (ptr, WEARENT, c);
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I wouldn`t enable '%s' for you]", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			else
			{
				opt->us = YES;
				opt->usq = EMPTY;
#ifdef DEBUG_TELNET
				if (ptr->debuglevel >= VERBOSE)
				{
					sprintf(tndmess, "[%s: I enabled '%s' for you]", PROGNAME, tnoptis(c));
					debug_write(ptr, tndmess, strlen(tndmess));
				}
#endif
			}
			break;
		case WANTYES:
			if (ptr->changeopt != NULL)
				(ptr->changeopt) (ptr, WEARE, c);
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You let me enable '%s']", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			if (opt->usq == EMPTY)
			{
				opt->us = YES;
			}
			else
			{
				opt->us = WANTNO;
				opt->usq = EMPTY;
				buff[1] = TCWONT;
				write_to_socket(ptr, buff, 3);
			}
			break;
		}						 /* end switch */
	else if (o == TCDONT)
		switch (opt->us)
		{
		case NO:
			/* already disabled */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= WARNINGS)
			{
				sprintf(tndmess, "[%s: I already have '%s' disabled]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			break;
		case YES:
			opt->us = NO;
			buff[1] = TCWONT;
			write_to_socket(ptr, buff, 3);
			if (ptr->changeopt != NULL)
				(ptr->changeopt) (ptr, WEARENT, c);
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: I disabled '%s' for you]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			break;
		case WANTNO:
			if (ptr->changeopt != NULL)
				(ptr->changeopt) (ptr, WEARENT, c);
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You let me disable '%s']", PROGNAME,
						tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			if (opt->usq == EMPTY)
			{
				opt->us = NO;
			}
			else
			{
				opt->us = WANTYES;
				opt->usq = EMPTY;
				buff[1] = TCWILL;
				write_to_socket(ptr, buff, 3);
			}
			break;
		case WANTYES:
			/* a length-two queue could save a potential request here... */
#ifdef DEBUG_TELNET
			if (ptr->debuglevel >= VERBOSE)
			{
				sprintf(tndmess, "[%s: You didn`t enable '%s' for me]", PROGNAME, tnoptis(c));
				debug_write(ptr, tndmess, strlen(tndmess));
			}
#endif
			opt->us = NO;
			if (opt->usq == OPPOSITE)
				opt->usq = EMPTY;
			break;
		}						 /* end switch */
	return 0;
}


/* Starting sub-negotiation */
static int cfsm_subbegin(struct tnstat *ptr, unsigned char c)
{
#ifdef DEBUG_TELNET
	if (ptr->debuglevel == PROGRAMMER)
	{
		sprintf(tndmess, "[%s: Getting sub-negotiation data]", PROGNAME);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	ptr->sbstate = TSB_START;
	ptr->sbbuffpos = 0;
	return 0;
}


/* Receiving telnet sub-negotiation data */
static int cfsm_subneg(struct tnstat *ptr, unsigned char c)
{
	struct fsm *p;
	int i;

	if (ptr->sbbuffpos >= TNSB_BUFFERSIZE-2)
	{
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= ERRORS)
		{
			sprintf(tndmess, "[%s: Too many chars in sub-negotiation data]", PROGNAME);
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		ptr->sbstate = TSSUBERR;
		return 1;
	}

	ptr->sbbuff[ptr->sbbuffpos++] = c;

	i = tnsbttable[ptr->sbstate][c];
	p = &tnsbmatrix[i];
	if (p->function != NULL)
		(void) (p->function) (ptr, c);
	ptr->sbstate = p->next;
	return 0;
}


/* Finished receiving sub-negotiation data. Time to deal with it. */
static int cfsm_subend(struct tnstat *ptr, unsigned char c)
{
#ifdef DEBUG_TELNET
#if	0
	int i, j;

	if (ptr->debuglevel == PROGRAMMER)
	{
		i = 0;
		j = ptr->sbbuff[0];
		if (j == TCWILL || j == TCWONT || j == TCDO || j == TCDONT)
		{
			sprintf(tndmess, "[%s]", tncomis(ptr->sbbuff[i]));
			debug_write(ptr, tndmess, strlen(tndmess));
			++i;
		}
		sprintf(tndmess, "[%s]", tnoptis(ptr->sbbuff[i]));
		debug_write(ptr, tndmess, strlen(tndmess));
		for (++i ; i < ptr->sbbuffpos; ++i)
		{
			sprintf(tndmess, "[%u]", (unsigned char)ptr->sbbuff[i]);
			debug_write(ptr, tndmess, strlen(tndmess));
		}
	}
#endif
#endif

#ifdef DEBUG_TELNET
	if (ptr->debuglevel == PROGRAMMER)
	{
		sprintf(tndmess, "[%s: End of sub-negotiation data]", PROGNAME);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	ptr->sbbuff[ptr->sbbuffpos++] = (unsigned char)TCIAC;
	ptr->sbbuff[ptr->sbbuffpos++] = (unsigned char)TCSE;

	switch (ptr->sbbuff[0])
	{
	case TNLineMode:
		sb_linemode(ptr);
		break;
	case TNNegotiateAboutWindowSize:
		sb_naws(ptr);
		break;
	default:
		break;
	}

	ptr->sbstate = -1;
	return 0;
}


/* Cleanup after error during receipt of sub-negotiation data */
static int sb_error(struct tnstat *ptr, unsigned char c)
{
#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: End of sub-negotiation after error]", PROGNAME);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	ptr->sbstate = -1;
	return 0;
}


/* Unsupported sub-negotiation option */
#if defined(__cplusplus) && !defined(DEBUG_TELNET)
static int sb_unsup(struct tnstat *, unsigned char)
#else
static int sb_unsup(struct tnstat *ptr, unsigned char c)
#endif
{
#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= WARNINGS)
	{
		sprintf(tndmess, "[%s: Unsupported sub-negotiation '%s']", PROGNAME, tnoptis(c));
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif
	return 0;
}


/* Subnegotiation of line mode */
static void sb_linemode(struct tnstat *ptr)
{
	switch (ptr->sbbuff[1])
	{
	case LM_MODE:
		sb_lm_mode(ptr);
		break;

	case LM_SLC:
		sb_lm_slc(ptr);
		break;

	/* If we didn't get one of the above,	*/
	/* it must be one of do/dont/will/wont	*/
	default:
		sb_lm_fwmask(ptr);
		break;
	}

	return;
}


void sb_lm_mode_set(struct tnstat *ptr, unsigned char mode)
{
	unsigned char buff[7];

	buff[0] = TCIAC;
	buff[1] = TCSB;
	buff[2] = TNLineMode;
	buff[3] = LM_MODE;
	buff[4] = mode;
	buff[5] = TCIAC;
	buff[6] = TCSE;
	write_to_socket(ptr, buff, sizeof(buff));

#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: Sent LINEMODE Mode %2x]", PROGNAME, mode);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif
}


/* Subnegotiation of linemode mode */
/* Additional checks need to be done here ~~~~~ */
static void sb_lm_mode(struct tnstat *ptr)
{
	unsigned char c;

	c = ptr->sbbuff[2];
#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: Got LINEMODE Mode %2x]", PROGNAME, c);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	if (c & LM_MODE_ACK)
	{
		ptr->lm.mode = c & ~LM_MODE_ACK;
		return;
	}

	ptr->lm.mode = c;
	sb_lm_mode_set(ptr, c | LM_MODE_ACK);
	return;
}


/* Subnegotiation of linemode forwardmask */
static void sb_lm_fwmask(struct tnstat *ptr)
{
	unsigned char c, buff[7];

#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: LINEMODE %s ForwardMask]",
				PROGNAME, tncomis( ptr->sbbuff[1] ));
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	c = ptr->sbbuff[1];
	if (c == (unsigned char)TCDO)
	{
		buff[0] = TCIAC;
		buff[1] = TCSB;
		buff[2] = TNLineMode;
		buff[3] = TCWONT;
		buff[4] = LM_FORWARDMASK;
		buff[5] = TCIAC;
		buff[6] = TCSE;
		write_to_socket(ptr, buff, sizeof(buff));
	}
	return;
}


/* Subnegotiation of linemode set local characters */
/* Not fully supported yet. ~~~~~ */
static void sb_lm_slc(struct tnstat *ptr)
{
	unsigned char *s, *buff;
	int i, j;
	int cnt;

	buff = malloc(ptr->sbbuffpos+2);
	if (buff == NULL)
		return;

	s = (unsigned char *)ptr->sbbuff;
	cnt = 0;
	j = 2;
	for (i = 2; i < ptr->sbbuffpos; i += 3)
	{
		if (ptr->sbbuff[i+1] != LM_SLC_NOSUPPORT)
		{
/* What do we need to do here? */
		}
		cnt += 3;
	}

	buff[0] = TCIAC;
	buff[1] = TCSB;
	buff[cnt] = TCIAC;
	buff[cnt+1] = TCSE;

#if	0
	if (cnt > 0)
	{
		write_to_socket(ptr, "\xff\xfa", 2);	/* Send IAC SB */
		write_to_socket(ptr, ptr->sbbuff, cnt + 4);
	}
#endif

#ifdef DEBUG_TELNET
	if (ptr->debuglevel == PROGRAMMER)
	{
		sprintf(tndmess, "[%s: Got %d bytes of SLC data]",
				PROGNAME, ptr->sbbuffpos - 4);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	return;
}


/* Subnegotiation of negotiate window size */
static void sb_naws(struct tnstat *ptr)
{
	int cols, rows;

	if (ptr->sbbuffpos != 7)
	{
#ifdef DEBUG_TELNET
		if (ptr->debuglevel == PROGRAMMER)
		{
			sprintf(tndmess, "[%s: ERROR - Got %d bytes of NAWS data]",
					PROGNAME, ptr->sbbuffpos - 4);
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		return;
	}

	cols = (ptr->sbbuff[1] << 8) + ptr->sbbuff[2];
	rows = (ptr->sbbuff[3] << 8) + ptr->sbbuff[4];

#ifdef DEBUG_TELNET
	if (ptr->debuglevel >= VERBOSE)
	{
		sprintf(tndmess, "[%s: NAWS -> cols=%u, rows=%u]",
				PROGNAME, cols, rows);
		debug_write(ptr, tndmess, strlen(tndmess));
	}
#endif

	if (ptr->naws_cb != NULL)
		(ptr->naws_cb) (cols, rows);

	return;
}


static void tnthemenable(struct tnstat *ptr, unsigned char c)
{
	unsigned char buff[3];
	struct tnopt *opt;

	opt = &(ptr->opt[c]);
	buff[0] = TCIAC;
	buff[1] = TCDO;
	buff[2] = c;

	switch (opt->them)
	{
	case NO:
		opt->them = WANTYES;
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= VERBOSE)
		{
			sprintf(tndmess, "[%s: Sent IAC DO '%s']", PROGNAME, tnoptis(c));
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		write_to_socket(ptr, buff, 3);
		break;
	case YES:
		/* already enabled */
		break;
	case WANTNO:
		if (opt->themq == EMPTY)
			opt->themq = OPPOSITE;
		/* else */
		/* already queued an enable request */
		break;
	case WANTYES:
		if (opt->themq == OPPOSITE)
			opt->themq = EMPTY;
		/* else */
		/* already negiotiating for enable */
		break;
	}
}


static void tnthemdisable(struct tnstat *ptr, unsigned char c)
{
	unsigned char buff[3];
	struct tnopt *opt;

	opt = &(ptr->opt[c]);
	buff[0] = TCIAC;
	buff[1] = TCDONT;
	buff[2] = c;

	switch (opt->them)
	{
	case NO:
		/* already disabled */
		break;
	case YES:
		opt->them = WANTNO;
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= VERBOSE)
		{
			sprintf(tndmess, "[%s: Sent IAC DONT '%s']", PROGNAME, tnoptis(c));
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		write_to_socket(ptr, buff, 3);
		break;
	case WANTNO:
		if (opt->themq == OPPOSITE)
			opt->themq = EMPTY;
		/* else */
		/* already negiotiating for disable */
		break;
	case WANTYES:
		if (opt->themq == EMPTY)
			opt->themq = OPPOSITE;
		/* else */
		/* already queued a disable request */
		break;
	}
}


static void tnusenable(struct tnstat *ptr, unsigned char c)
{
	unsigned char buff[3];
	struct tnopt *opt;

	opt = &(ptr->opt[c]);
	buff[0] = TCIAC;
	buff[1] = TCWILL;
	buff[2] = c;

	switch (opt->us)
	{
	case NO:
		opt->us = WANTYES;
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= VERBOSE)
		{
			sprintf(tndmess, "[%s: Sent IAC WILL '%s']", PROGNAME, tnoptis(c));
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		write_to_socket(ptr, buff, 3);
		break;
	case YES:
		/* already enabled */
		break;
	case WANTNO:
		if (opt->usq == EMPTY)
			opt->usq = OPPOSITE;
		/* else */
		/* already queued an enable request */
		break;
	case WANTYES:
		if (opt->usq == OPPOSITE)
			opt->usq = EMPTY;
		/* else */
		/* already negiotiating for enable */
		break;
	}
}


static void tnusdisable(struct tnstat *ptr, unsigned char c)
{
	unsigned char buff[3];
	struct tnopt *opt;

	opt = &(ptr->opt[c]);
	buff[0] = TCIAC;
	buff[1] = TCWONT;
	buff[2] = c;

	switch (opt->us)
	{
	case NO:
		/* already disabled */
		break;
	case YES:
		opt->us = WANTNO;
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= VERBOSE)
		{
			sprintf(tndmess, "[%s: Sent IAC WONT '%s']", PROGNAME, tnoptis(c));
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		write_to_socket(ptr, buff, 3);
		break;
	case WANTNO:
		if (opt->usq == OPPOSITE)
			opt->usq = EMPTY;
		/* else */
		/* already negiotiating for disable */
		break;
	case WANTYES:
		if (opt->usq == EMPTY)
			opt->usq = OPPOSITE;
		/* else */
		/* already queued a disable request */
		break;
	}
}


/* change default settings for individual telnet options */
void tnsetopt(struct tnstat *tn, enum tnoption opt, enum tnoset choice)
{
	struct tnopt *p;

	assert(tn != NULL);

	p = &(tn->opt[opt]);
	switch (choice)
	{
	case WEARE:
		tnusenable(tn, opt);
		break;
	case WEARENT:
		tnusdisable(tn, opt);
		break;
	case WEWOULD:
		p->weshould = 1;
		break;
	case WEWOULDNT:
		p->weshould = 0;
		break;
	case THEYARE:
		tnthemenable(tn, opt);
		break;
	case THEYARENT:
		tnthemdisable(tn, opt);
		break;
	case THEYSHOULD:
		p->theyshould = 1;
		break;
	case THEYSHOULDNT:
		p->theyshould = 0;
		break;
	default:
		break;
	}
}


/* change default settings for individual telnet options */
int tngetopt(struct tnstat *tn, enum tnoption opt, enum tnoset choice)
{
	struct tnopt *p;

	assert(tn != NULL);

	p = &(tn->opt[opt]);
	switch (choice)
	{
	case WEARE:
		return p->us ? 1 : 0;
	case WEARENT:
		return p->us ? 0 : 1;
	case WEWOULD:
		return p->weshould ? 1 : 0;
	case WEWOULDNT:
		return p->weshould ? 0 : 1;
	case THEYARE:
		return p->them ? 1 : 0;
	case THEYARENT:
		return p->them ? 0 : 0;
	case THEYSHOULD:
		return p->theyshould ? 1 : 0;
	case THEYSHOULDNT:
		return p->theyshould ? 0 : 1;
	default:
		return -1;
	}
}


void tnsynch(struct tnstat *ptr)
{
	if (ptr->synch == 0)
	{
#ifdef DEBUG_TELNET
		if (ptr->debuglevel >= VERBOSE)
		{
			sprintf(tndmess, "[%s: START SYNCH]", PROGNAME);
			debug_write(ptr, tndmess, strlen(tndmess));
		}
#endif
		ptr->synch = 1;
	}
}


#ifdef	NEW_SOCKET_IO

/*** perform_socket_write for all Non-Windows platforms ***/
static int perform_socket_write(struct tnstat *ptr, unsigned char *str, size_t length)
{
	ssize_t result;

	result = write(ptr->sock, str, length);

	if (result > 0)
	{	/* Write was successful. */
		return result;
	}
	if (result == 0)
	{	/* This should never happen! */
#ifdef DEBUG_TELNET
		sprintf(tndmess, "SYSERR: Huh??  write() returned 0???  Please report this!");
		debug_write(ptr, tndmess, strlen(tndmess));
#endif
		return -1;
	}

	/* result < 0, so an error was encountered - is it transient?	*/
	/* Unfortunately, different systems use different constants to	*/
	/* indicate this.												*/

#ifdef EAGAIN       /* POSIX */
	if (errno == EAGAIN)
	return 0;
#endif

#ifdef EWOULDBLOCK  /* BSD */
	if (errno == EWOULDBLOCK)
		return 0;
#endif

#ifdef EDEADLK      /* Macintosh */
	if (errno == EDEADLK)
		return 0;
#endif

	/* Looks like the error was fatal.  Too bad. */
	return -1;
}


/*
 * write_to_socket takes a socket, and text to write to the
 * socket.  It keeps calling the system-level write() until all
 * the text has been delivered to the OS, or until an error is
 * encountered.
 *
 * Returns:
 *  0  If all is well and good,
 * -1  If an error was encountered, so that the player should be cut off
 */
/*** This routine sends a string of characters down socket ***/
static int write_to_socket(struct tnstat *ptr, unsigned char *txt, int bytes_to_write)
{
	ssize_t bytes_written;
	int sock;

	sock = ptr->sock;

	while (bytes_to_write > 0)
	{
		bytes_written = perform_socket_write(ptr, txt, bytes_to_write);

		if (bytes_written < 0)
		{	/* Fatal error.  Disconnect the player. */
#ifdef DEBUG_TELNET
			sprintf(tndmess, "Error writing to socket %d", sock);
			debug_write(ptr, tndmess, strlen(tndmess));
#endif
			return -1;
		}
		if (bytes_written == 0)
		{
			/*
			* Temporary failure -- socket buffer full.  For now we'll just
			* cut off the player, but eventually we'll stuff the unsent
			* text into a buffer and retry the write later.  JE 30 June 98.
			*/
#ifdef DEBUG_TELNET
			sprintf(tndmess, "write_to_socket: socket %d would block, about to close", sock);
			debug_write(ptr, tndmess, strlen(tndmess));
#endif
			return -1;
		}

		txt += bytes_written;
		bytes_to_write -= bytes_written;

#ifdef DEBUG_TELNET		/* ~~~~~ */
if (bytes_to_write > 0)
{
	sprintf(tndmess, "write_to_socket: short write for socket %d", sock);
	debug_write(ptr, tndmess, strlen(tndmess));
}
#endif
	}

	return 0;
}

#endif  /* NEW_SOCKET_IO */


#ifdef DEBUG_TELNET
/* debug wrap around for write_to_socket() */
void debug_write(struct tnstat *ptr, char *buff, int size)
{
#if	1
write_to_socket(ptr, (unsigned char *)"\n", 1);
#endif
	if (ptr->debugstream == NULL)
		write_to_socket(ptr, (unsigned char *)buff, size);
	else
	{
		fputs(buff, ptr->debugstream);
		fflush(ptr->debugstream);
	}
}
#endif

