/*****************************************************************************

        Neils Unix Talk Server (NUTS) - (C) Neil Robertson 1992-1995
            Last update: 12th January 1995   Version 2.3.0
  
  This version now works with character mode clients.
  
  Feel free to modify to code in any way but remember the original copyright
  is mine - this means you can't sell it or pass it off as your own work!
  This INCLUDES modified source. If wish to sell something don't include ANY
  of my code in whatever it is unless you ask my permission first.
     If and when you make any changes to the code please leave the original 
  copyright in the code header if you wish to distribute it. For all the
  doubting Thomases out there yes I DO have the copyright to this - automatic
  copyright to any written work or computer software applies in the UK where
  this was written (unlike the States where you have to send something off 
  I think).
  
  Thanks to:
     Darren Seryck - who thought up the name NUTS.
     Simon Culverhouse - for being the bug hunter from hell.
     Steve Guest - my networks lecturer at Loughborough University.
     Dave Temple - the hassled network admin at above uni who told me about
                   the select() function and how to use it.
     Satish Bedi - (another hassled admin) for listening so understandingly 
                   while I explained to him why the comp sci development 
                   machine had to be rebooted for the 3rd time that week.
     The 1992/1993 LUTCHI team - for coming up with the search command idea 
                                 (knew they were good for something :-) )
     Tim Bernhardt - for the internet name resolution code.
     Sven Barzanallana - for some bug fixes and 2.1.1.
     An HP-UX network programming manual - for providing some "help" >:-)
     Trent 96.2 FM - for providing some good music to work to at uni.
     RTC - for giving me a job.
     Anyone who has ever used NUTS.
  
  This program (or its ancestor at any rate) was originally a university 
  project and first went live on the net in November 1992 as Hectic House. 
  Since then it has spread and spread (bit like flu really). :-)
  
  Bug reports, gripes, whines, ideas:
  	Neil Robertson 
  	neil@realtime.demon.co.uk  
     (until I get sacked for spending too much time on this and not on my work
      then it'll be neil@cardboard_box.under.bridge)
  
*****************************************************************************/

#include <stdio.h>
#define __USE_BSD   1
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

/* File definitions */
#define	INITFILE	"init_data"
#define	SNAPSHOT_FILE INITFILE ".snapshot"

/* Other definitions */
#define MIN_PORT		1024U	/* Absolute minimum value for port number */
#define MAX_PORT		32767U	/* Absolute maximum value for port number */
								/* Must be convertable to a postive int   */
#ifndef FD_SETSIZE
  #define FD_SETSIZE	256
#endif
#define ABS_MAX_USERS	FD_SETSIZE-1 /* Absolute max number of users allowed */
#define MAX_AREAS		200		/* Set a reasonable limit on number of areas */
#define TOPIC_LEN		60
#define DESC_LEN		35
#define NAME_LEN		15
#define	EMAIL_LEN		60		/* Length of e-mail address string */
#define	ONOFF_LEN		60		/* Length of log-on and log-off messages */
#define	ENTER_EXIT_LEN	60		/* Length of area enter/exit messages */
#define	MACRO_LEN		60		/* Length of user defined macro strings */
#define AREA_LABEL_LEN	5
#define AREA_NAME_LEN	15
#define AREA_STATUS_LEN	11		/* Length of longest string for area status */
#define SITE_LEN		80

#define	MIN_COLS		40		/* minimum number of screen columns */
#define	MAX_COLS		255		/* maximum number of screen columns */
#define	DEFAULT_COLS	80		/* default number of screen columns */
#define	MIN_ROWS		10		/* minimum number of screen rows */
#define	MAX_ROWS		100		/* maximum number of screen rows */
#define	DEFAULT_ROWS	24		/* default number of screen rows */

/* Values used to indicate status of areas and their accessability */
/* Do not change the order or values of the following #defines!	   */
#define AREA_STATUS_FIXED		0	/* Area is always public */
#define AREA_STATUS_PUBLIC		1	/* Area is currently public */
#define AREA_STATUS_PRIVATE		2	/* Area is currently private */
#define AREA_STATUS_WIZ_FIXED	3	/* Area is fixed and for wiz only */
#define AREA_STATUS_WIZ_PUBLIC	4	/* Area is public and for wiz only */
#define AREA_STATUS_WIZ_PRIVATE	5	/* Area is private and for wiz only */
#define AREA_STATUS_JAIL		6	/* Area is the jail */


char *days[] = { "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun", NULL };

char *months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
				   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL };

char *link_labels = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

/* Area structure */
typedef struct
{
	char label[AREA_LABEL_LEN];
	char name[AREA_NAME_LEN];
	char topic[TOPIC_LEN];

	int status;					/* Indicates the status of the area */
	char *move;					/* List of areas reachable from this one */

	unsigned int wiz_only: 1;	/* Set if only wiz and up allowed in area */
	unsigned int searchable: 1;	/* Set if non-wiz can search areas msesages */
	unsigned int linked: 1;     /* Set if the areas message board is linked */
} AREA_DATA;

AREA_DATA *astr;
int num_areas;

int num_of_users;				/* A count of the number of users logged in */

/* Prototypes for miscellaneous functions */
int init_globals(void);
int read_init_data(void);
int verify_init_data(void);
int get_com_index(char *);
void do_snapshot(void);

/* User level definitions */
#define DUNCE	0
#define USER	1
#define WIZARD	2
#define SU		3
#define SU_SU	4
#define NOONE	-1			/* Used when no users are allowed to do something */
#define MAGICAL WIZARD		/* Users of this level and up have extra features */

#define	MAX_LEVEL	SU_SU	/* Absolute maximum user level */
#define	CLL			SU		/* Level of user who can log in to a closed port */
#define AREA_MODIFY_LEVEL	SU_SU	/* Level of user who can modify areas */

#define LEVSTR_LEN	8		/* Maximum allowed length of level strings */

char *default_levstr[] = { "DUNCE", "USER", "WIZARD", "SU", "*SU*" };
char *levstr[MAX_LEVEL + 2];


char *areas_status[] = { "fixed", "public", "private", "wiz_fixed",
					 		"wiz_public", "wiz_private", "jail", NULL };
char *yesno[] = { "NO", "YES", NULL };
char *onoff[] = { "OFF", "ON", NULL };

char *syserror = "Sorry - a system error has occurred";

int jail = -1;			/* Holds the number of the area used for the jail */

unsigned int port;
int max_areas;			/* Maximum number of areas allowed */
int	max_users;			/* This must NOT be greater than FD_SETSIZE - 1 */

int max_attempts;		/* maximum allowed number of login attempts */

int system_open;		/* Are users of any level allowed to login? */
int allow_new;			/* Are new users allowed? */
int min_login_lvl;		/* Minimum level of users allowed to login to this port */
int init_user_lvl;		/* Default level for users logging in for the first time */
int no_logout_lvl;		/* Minimum level for users who can not be auto-logged out */
int login_show_lvl;		/* Enable showing of user level before 'SIGN ON' */
int login_show_min;		/* Minimum level that will be shown at login */

int profile_lines;		/* Number of profile lines user can store */
int max_area_lines;		/* Max. lines to hold in area conversation buffer */
int max_echo_lines;		/* Max. lines to hold in echo buffer */
int max_tell_lines;		/* Max. number of tells to record keep per user */
int max_log_lines;		/* Max. number of login/logouts to record */
int max_wiz_lines;		/* Max. number of wizard tell/emotes to record */

int atmos_on;			/* Enable display of area atmosphere messages */
int allow_bold;			/* Allow the use of ^ to mark text to show in bold */
int prevent_fakes;		/* Prevent fake system messages and messages from other users */
int show_examines;		/* Tell a user which other user .examined them */
int show_whispers;		/* Show as whispering who is sending tells to whom */
int show_ignores;		/* Show everyone message types user will ignore/see */
int show_user_ign;		/* Tell a user they are being ignored */

/* The next items are used in check_timeout(). Times are in seconds */
int heartbeat;			/* alarm time in seconds (must not exceed 60) */
int login_timeout;		/* login time out - can't be less than heartbeat */
int idle_mention;		/* Time (in minutes) before eyes glaze over */
int idle_warning;		/* Time (in minutes) to start auto-logout warnings */
int idle_logout;		/* Time (in minutes) to log out an idle user */
int message_life;		/* Message lifetime in days */

int syslog_on;			/* Enable logging of various events */

/* Information regarding global variables and parsing */
/* of the file containing the initialization data.	  */

#define	INIT_HEADER		"init:"
#define	AREAS_HEADER	"areas:"
#define	TOPICS_HEADER	"topics:"
#define	USERS_HEADER	"users:"
char *section_heads[] = { INIT_HEADER, AREAS_HEADER, TOPICS_HEADER, USERS_HEADER, NULL };

enum VAR_TYPE { Bool, Int, Uint, Level, Levstr };

typedef struct
{
	unsigned int	level;
	char *word;
	void *var;
	enum VAR_TYPE	var_type;
	int  min;
	int  max;
	int  init_value;
} GLOBAL_DATA;

/* 0 is allowed here for port number as an indication that	*/
/* port number has not yet been set to a valid value.		*/
GLOBAL_DATA	global_data[] = {
	{NOONE, "port",				&port,			Uint, 0, MAX_PORT, 0 },
	{NOONE, "areas",			&max_areas,		Int,  0, MAX_AREAS, MAX_AREAS },
	{SU_SU, "users",			&max_users,		Int,  1,ABS_MAX_USERS, 35 },
	{SU_SU, "attempts",			&max_attempts,	Int,  1,  10,	3 },
/* Do NOT change the CLL level reference here. Instead change its #define */
	{  CLL, "open_system",		&system_open,	Bool, 0,   1,	1 },
	{	SU, "new_users",		&allow_new,		Bool, 0,   1,	1 },
	{SU_SU, "min_login_level",	&min_login_lvl, Level,	DUNCE,	SU,   USER },
	{SU_SU, "init_user_level",	&init_user_lvl, Level,	DUNCE,	USER, USER },
	{SU_SU, "no_logout_level",	&no_logout_lvl, Level,	DUNCE,	SU,   SU },
	{SU_SU, "login_show_level",	&login_show_lvl,Bool, 0,   1,   1 },
	{SU_SU, "login_show_min",	&login_show_min,Level,	DUNCE,	SU,	  WIZARD },
	{SU_SU, "profile_lines",	&profile_lines,	Int,  0, 100,  15 },
	{SU_SU, "area_lines",		&max_area_lines,Int,  0, 100,  20 },
	{SU_SU, "echo_lines",		&max_echo_lines,Int,  0, 100,  20 },
	{SU_SU, "tell_lines",		&max_tell_lines,Int,  0, 100,  20 },
	{SU_SU, "log_lines",		&max_log_lines,	Int,  0, 100,  20 },
	{SU_SU, "wiz_lines",		&max_wiz_lines,	Int,  0, 100,  20 },
	{	SU, "atmosphere",		&atmos_on,		Bool, 0,   1,   1 },
	{SU_SU, "prevent_fakes",	&prevent_fakes,	Bool, 0,   1,   1 },
	{SU_SU, "bold_allowed",		&allow_bold,	Bool, 0,   1,   1 },
	{	SU, "whispers",			&show_whispers,	Bool, 0,   1,   0 },
	{	SU, "ignores",			&show_ignores,	Bool, 0,   1,   0 },
	{	SU, "user_ignores",		&show_user_ign,	Bool, 0,   1,   0 },
	{	SU, "examines",			&show_examines,	Bool, 0,   1,   0 },
	{NOONE, "heartbeat",		&heartbeat,		Int, 15,  60,  30 },	/* Main timer (in seconds) */
	{NOONE, "timeout_login",	&login_timeout,	Int, 30, 600, 180 },	/* A multiple of heartbeat */
	{SU_SU, "idle_mention",		&idle_mention,	Int,  0,  30,   5 },	/* Time for eyes to glaze */
	{SU_SU, "idle_warn",		&idle_warning,	Int,  0,  60,  15 },	/* Warn of auto logout in x minutes */
	{SU_SU, "idle_logout",		&idle_logout,	Int,  0,  60,  30 },    /* Log-out after x minutes inactivity */
	{SU_SU, "lifetime",			&message_life,	Int,  1,  30,   5 },	
	{SU_SU, "logging",			&syslog_on,		Bool, 0,   1,   1 },	
	{SU_SU, "level0",			NULL,		 Levstr,  0,   0,   0 },	
	{SU_SU, "level1",			NULL,		 Levstr,  1,   1,   1 },	
	{SU_SU, "level2",			NULL,		 Levstr,  2,   2,   2 },	
	{SU_SU, "level3",			NULL,		 Levstr,  3,   3,   3 },	
	{SU_SU, "level4",			NULL,		 Levstr,  4,   4,   4 },	
	{SU_SU, NULL,				NULL,			Int, -1,  -1,   0 }
};


int main(void)
{
	puts("Initializing global variables...");
	if (init_globals())
	{
		puts("Error initializing global variables\n");
		exit(1);
	}

	puts("Reading old initialization data file...");
	if ( read_init_data() )
		exit(1);

	puts("Verifying initialization data...");
	if ( verify_init_data() )
		exit(1);

	puts("Creating snapshot file...");
	do_snapshot();

	puts("Finished conversion of initialization data file.\n");

	return 1;
}


/* Force a string to all upper case */
void strupcase(char *string)
{
int i;

	for (i = 0; string[i] != '\0'; ++i)
		string[i] = toupper(string[i]);
}


/*** Initialize various global variables ***/
int init_globals(void)
{
int error;
int i, j;
int *var;

	error = 0;

	for (i = 0; global_data[i].word != NULL; ++i)
	{
		if (global_data[i].var_type == Levstr)
		{
			j = global_data[i].init_value;
			levstr[j] = strdup(default_levstr[j]);
			strupcase( levstr[j] );
		}
		else
		{
			var = global_data[i].var;

			/* Check the limits on the variable */
			if (global_data[i].min > global_data[i].max)
			{
				fprintf(stderr, "init_globals - '%s' has invalid min or max value\n",
						global_data[i].word);
				error = 1;
			}

			/* Check the initial (default) value to make sure it is legal */
			if (global_data[i].init_value < global_data[i].min ||
				global_data[i].init_value > global_data[i].max)
			{
				fprintf(stderr, "init_globals - '%s' has bad initial value\n",
						global_data[i].word);
				error = 1;
			}

			*var = global_data[i].init_value;
		}
	}

	levstr[MAX_LEVEL + 1] = NULL;
	num_areas = 0;

	return error;
}


/*** return pos. of second word in str ***/
char *skip_first(str)
char *str;
{
    char *pos = str;

    while (*pos <= ' ' && *pos)
        ++pos;
    while (*pos > ' ')
        ++pos;
    while (*pos <= ' ' && *pos)
        ++pos;
    return pos;
}


/*** parse a string from an buffer ***/
/* This function returns the first word from the */
/* input buffer in the output buffer limited to  */
/* the number of characters specified by max_len.*/
/* The returned string is nul terminated.        */
void get_word(input, output, max_len)
char *input, *output;
int max_len;
{
int len;

    /* Copy one less character than the size of the buffer */
    for (len = 1; len < max_len; ++len)
    {
        if (isspace(*input))
            break;
        *output++ = *input++;
    }

    *output = '\0';
}


/*** determine area number based on link code ***/
int find_area_num(char link_code)
{
	int i;

	for (i = 0; i < num_areas; ++i)
	{
		if (link_labels[i] == link_code)
			return i+1;
	}

	return(num_areas);
}


/*** read in initialize data ***/
int read_init_data(void)
{
	char line[80];
	char *s;
	int a;
	FILE *fp;

	printf("Reading initialization data from file %s\n", INITFILE);
	if ((fp = fopen(INITFILE, "r")) == NULL)
	{
		perror("\nNUTS: Couldn't read init file");
		return(1);
	}

	fgets(line, sizeof(line), fp);

/* read in important system data & do a check of some of it */
  /* Line 1: <port number> <number of areas> <atmos on> \
  **         <syslog on> <newusers on> <message life time - days> \
  ** [ the next bit is the EXTENDED format ]
  **         <max login trys> <login time out - seconds> \
  **         <eyes glaze time - mins> \
  **         <normal warning - mins> <normal time out - mins> \
  **         <afk warning - mins> <afk time out - mins>
  ** Note: The toggle options are: 1, yes, y, on
  **                               0, no,  n, off
  */
	atmos_on = -1;
	syslog_on = -1;
	message_life = -1;
	allow_new = -1;
	sscanf(line, "%d %d %d %d %d %d",
				&port, &num_areas, &atmos_on, &syslog_on, &allow_new, &message_life);
	if (port < 1 || port > 65535 || num_areas > MAX_AREAS || atmos_on < 0 || atmos_on > 1 || syslog_on < 0 || syslog_on > 1 || message_life < 1 || allow_new < 0 || allow_new > 1)
	{
		perror("\nNUTS: Error in init file on line 1\n");
		fclose(fp);
		return(1);
	}

	/*** Allocate storage for all the rooms ***/
	if ((astr = (AREA_DATA *) malloc(sizeof(AREA_DATA) * num_areas)) == NULL)
	{
		fprintf(stderr, "Error allocating storage for rooms\n");
		fclose(fp);
		return(1);
	}
	for (a = 0; a < num_areas; ++a)
	{
		memset(&astr[a], 0, sizeof(AREA_DATA));
		astr[a].searchable = 1;

		if ((astr[a].move = (char *)malloc(sizeof(char) * num_areas)) == NULL)
		{
			fprintf(stderr, "Error allocating storage for rooms\n");
			fclose(fp);
			return(1);
		}
	}

/* read in area descriptions and joinings */
	for (a = 0; a < num_areas; ++a)
	{
		fgets(line, sizeof(line), fp);

		/* Handle any special indicators at start of line */
		s = line;
		while (!isalpha(*s))
		{
			switch (*s)
			{
			/* If 1 don't allow users access to message board */
			/* If 2 don't allow access to the room either     */
			case '-':
				astr[a].searchable = 0;
				if (s[1] == '-')
				{
					astr[a].wiz_only = 1;
					++s;
				}
				break;

			case '*':	/* Is this room to be used as the jail? */
				if (jail >= 0)
				{
					perror("\nNUTS: More than one room specified as being the jail");
					fclose(fp);
					return(1);
				}
				jail = a;
				break;

			case '!':
				astr[a].linked = 1;
				break;

			case '$':
				fprintf(stderr, "Warning: $ flag in init file on line %d ignored\n", a + 2);
				break;
			}

			++s;
		}

		sprintf(astr[a].label, "%d", a+1);
		get_word(s, astr[a].name, AREA_NAME_LEN);
		s = skip_first(s);
		get_word(s, astr[a].move, num_areas);
		s = skip_first(s);
		sscanf(s, "%d", &astr[a].status);

		if (!astr[a].name[0] || !astr[a].move[0] || astr[a].status < 0 || astr[a].status > 2)
		{
			fprintf(stderr, "\nNUTS: Error in init file on line %d\n", a + 2);
			fclose(fp);
			return(1);
		}

		if (astr[a].status < 2)
			astr[a].status ^= 1;	/* Fixed/public is flipped in old files */
	}
	fclose(fp);

	return 0;
}


/* Perform additional tests on global variables after init_data	*/
/* file has been read to look for additional errors or to make	*/
/* adjustments to any of the values that were read in.			*/
int verify_init_data(void)
{
int error;

	error = 0;

	/* Check some things which can only be set in initialization data file */
	if (port < MIN_PORT || port > MAX_PORT)
	{
		fprintf(stderr, "Port number is invalid or was not specified\n");
		error = 1;
	}

	if (jail < 0)
	{
		fprintf(stderr, "No area was specified as the jail. Using %s\n",
				astr[0].name);
		jail = 0;
	}	

	/* Now check on some things which can be modified */
	/* at any time via the use of the .system command */

	if (allow_new)
	{
		if (min_login_lvl > init_user_lvl)
			allow_new = 0;
	}

	if (idle_logout == 0)
	{
		if (idle_warning > 0)
		{
			idle_warning = 0;

			fprintf(stderr, "Auto-logout is disabled - ignoring idle warning time.\n");
			error = 1;
		}
	}
	else
	{
		/* Entered value is how many minutes before auto-logout should the */
		/* talker start warning a user that they have been idling too much */
		/* which is later converted in to when warnings should start.	   */
		if (idle_warning == 0)
		{
			if (idle_warning > 0 && idle_mention > idle_logout)
			{
				fprintf(stderr, "idle_mention must be less than auto-logout time\n");
				error = 1;
			}
		}
		else
		{
			if (idle_warning >= idle_logout)
			{
				fprintf(stderr, "idle_warning must be less than the auto-logout time\n");
				error = 1;
			}
			else
			{
				/* Now its safe to convert idle_warning to when warnings should start */
				idle_warning = idle_logout - idle_warning;

				if (idle_warning < idle_mention)
				{
					fprintf(stderr, "idle_warning is too large - current maximum is %d.\n",
							idle_logout - idle_mention);
					error = 1;
				}
			}
		}
	}

	return error;
}


void snapshot_init_section(FILE *fp)
{
int i;
int *var;

	fprintf(fp, "%s\n", INIT_HEADER);

	/* Output strings which specify user levels */
	for (i = 0; i < MAX_LEVEL; ++i)
		fprintf(fp, "level%d %s\n", i, levstr[i]);

	/* Output global initialization data */
	for (i = 0; global_data[i].word != NULL; ++i)
	{
		var = global_data[i].var;
		if (global_data[i].var == NULL)	/* Don't output level strings */
			continue;

		fprintf(fp, "%s ", global_data[i].word);

		switch (global_data[i].var_type)
		{
		case Bool:
			fprintf(fp, "%s\n", yesno[*var]);
			break;

		case Int:
			fprintf(fp, "%d\n", (int)*var);
			break;

		case Uint:
			fprintf(fp, "%u\n", (unsigned int)*var);
			break;

		case Level:
			fprintf(fp, "%s\n", levstr[*var]);
			break;

		case Levstr:	/* Level strings have already been output */
			break;
		}
	}

	fprintf(fp, "\n");
}


void snapshot_areas_section(FILE *fp)
{
AREA_DATA *area;
int area_node;
int move_node;
int link_num;
int status;

	fprintf(fp, "%s\n", AREAS_HEADER);

	for (area_node = 0; area_node < num_areas; ++area_node)
	{
		area = &astr[area_node];
		fprintf(fp, "%s %s ", area->label, area->name);
		if (area_node == jail)
			status = AREA_STATUS_JAIL;
		else
		{
			status = area->status;
			if (area->wiz_only)
				status += AREA_STATUS_WIZ_FIXED;
		}
		fprintf(fp, "%s %c",
				areas_status[status],
				tolower(yesno[area->searchable][0]));

		for (move_node = 0; move_node < num_areas; ++move_node)
		{
			link_num = find_area_num(area->move[move_node]);
			if (area->move[move_node] && area_node != link_num)
			{
				fprintf(fp, " %d", link_num);
			}
		}

		fprintf(fp, "\n");
	}

	fprintf(fp, "\n");
}


void snapshot_topics_section(FILE *fp)
{
int area_node;
AREA_DATA *area;

	fprintf(fp, "%s\n", TOPICS_HEADER);

	for (area_node = 0; area_node < num_areas; ++area_node)
	{
		area = &astr[area_node];

		/* Only output a topic line if area has a topic */
		if (area->topic[0] != '\0')
			fprintf(fp, "%s %s\n", area->label, area->topic);
	}

	fprintf(fp, "\n");
}


/*** Create a snapshot of the current state of the talker. ***/
/*** This includes all values shown in system status, area ***/
/*** information and topics. This information is written to***/
/*** a file which could be used as a new init_data file.   ***/
/*** This routine can be called from the alarm timer code. ***/
void do_snapshot(void)
{
FILE *fp;

	fp = fopen(SNAPSHOT_FILE, "w");
	if (fp == NULL)
	{	
		fprintf(stderr, "Unable to create snapshot file\n");
		return;
	}

	snapshot_init_section(fp);
	snapshot_areas_section(fp);
	snapshot_topics_section(fp);
	fclose(fp);
}
